<?php
// This file is part of Moodle Course Rollover Plugin
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.


/**
 * @package     local_schoolinfo
 * @author      iProgrammer
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_schoolinfo;

use dml_exception;
use stdClass;

class manager {

	/** Insert the data into our database table.
	 *
	 * @param string $school_name
	 * @param string $school_udise
	 * @return bool true if successful
	 */
	public function create_schoolinfo( string $school_name, string $school_board, string $school_district, string $school_udise, string $school_pincode ): bool {
		global $DB;
		$record_to_insert                 = new stdClass();
		$record_to_insert->schoolname     = $school_name;
		$record_to_insert->schoolboard    = $school_board;
		$record_to_insert->schooldistrict = $school_district;
		$record_to_insert->schooludise    = $school_udise;
		$record_to_insert->schoolpincode  = $school_pincode;
		try {
			return $DB->insert_record( 'local_schoolinfo', $record_to_insert, false );
		} catch ( dml_exception $e ) {
			return false;
		}
	}

	/** Get a single schoolinfo from its id.
	 *
	 * @param int $schoolinfoid the schoolinfo we're trying to get.
	 * @return object|false schoolinfo data or false if not found.
	 */
	public function get_schoolinfo( int $schoolinfoid ) {
		global $DB;
		return $DB->get_record( 'local_schoolinfo', array( 'id' => $schoolinfoid ) );
	}

	/** Update details for a single schoolinfo.
	 *
	 * @param int    $schoolinfoid the schoolinfo we're trying to get.
	 * @param string $school_name the new text for the schoolinfo.
	 * @param string $school_udise the new type for the schoolinfo.
	 * @return bool schoolinfo data or false if not found.
	 */
	public function update_schoolinfo( int $schoolinfoid, string $school_name, string $school_board, string $school_district, string $school_udise, string $school_pincode ): bool {
		global $DB;
		$object                 = new stdClass();
		$object->id             = $schoolinfoid;
		$object->schoolname     = $school_name;
		$object->schoolboard    = $school_board;
		$object->schooldistrict = $school_district;
		$object->schooludise    = $school_udise;
		$object->schoolpincode  = $school_pincode;
		return $DB->update_record( 'local_schoolinfo', $object );
	}

	/** Update the type for an array of schoolinfos.
	 *
	 * @return bool schoolinfo data or false if not found.
	 */
	public function update_schoolinfos( array $schoolinfoids, $type ): bool {
		global $DB;
		list($ids, $params) = $DB->get_in_or_equal( $schoolinfoids );
		return $DB->set_field_select( 'local_schoolinfo', 'schooludise', $type, "id $ids", $params );
	}

	/** Delete a schoolinfo and all the read history.
	 *
	 * @param $schoolinfoid
	 * @return bool
	 * @throws \dml_transaction_exception
	 * @throws dml_exception
	 */
	public function delete_schoolinfo( $schoolinfoid ) {
		global $DB;
		$transaction    = $DB->start_delegated_transaction();
		$deletedMessage = $DB->delete_records( 'local_schoolinfo', array( 'id' => $schoolinfoid ) );
		if ( $deletedMessage ) {
			$DB->commit_delegated_transaction( $transaction );
		}
		return true;
	}

	/** Delete all schoolinfos by id.
	 *
	 * @param $schoolinfoids
	 * @return bool
	 */
	public function delete_schoolinfos( $schoolinfoids ) {
		global $DB;
		$transaction        = $DB->start_delegated_transaction();
		list($ids, $params) = $DB->get_in_or_equal( $schoolinfoids );
		$deletedMessages    = $DB->delete_records_select( 'local_schoolinfo', "id $ids", $params );
		if ( $deletedMessages ) {
			$DB->commit_delegated_transaction( $transaction );
		}
		return true;
	}
}
